# Copyright (C) 2025 Arturs Ontuzans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import bpy

class UI_PT_Rig_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Rig_Panel"
    bl_label = "UE Vehicle Base Rigging"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"

    def draw(self, context):
        layout = self.layout

        scene = context.scene
        
        layout.label(text = "Vehicle rigging", icon = 'OUTLINER_OB_ARMATURE')

        if scene.dynamic_wheel_count is True:
            self.wheel_prop('Vehicle Body', 'vehicle_body', scene, True)

            row = layout.row()
            row.operator('view3d.add_another_wheel', text = "Add Wheel", icon="ADD")
            for index, item in enumerate(scene.multiple_wheels):
                self.n_wheel_prop(item, index, scene)
        
        else:
            self.wheel_prop('Vehicle Body', 'vehicle_body', scene, True)
            self.wheel_prop('FR', 'wheel_FR', scene, False)
            self.wheel_prop('RR', 'wheel_RR', scene, False)
            self.wheel_prop('FL', 'wheel_FL', scene, False)
            self.wheel_prop('RL', 'wheel_RL', scene, False)

        row = layout.row()
        row.prop(scene, 'dynamic_wheel_count')

        row = layout.row()
        row.prop(scene, 'rig_calipers')

        if scene.rig_calipers is True:
            row = layout.row()
            row.prop(scene, 'wheels_as_parents_of_calipers')

        row = layout.row()
        row.prop(scene, 'body_separate_bone')

        row = layout.row()
        row.prop(scene, 'bone_length')

        row = layout.row()
        row.operator('view3d.rig_vehicle')

        row = layout.row(align = True)
        row.label(text = "Export", icon = 'EXPORT')

        grid = layout.grid_flow(columns = 2, align = True)
        grid.operator('view3d.export_vehicle_rig_fbx')
        grid.operator('view3d.export_vehicle_rig_gltf')

    def wheel_prop(self, wheel_name, prop_name, scene, is_vehicle_body):
        box = self.layout.box()
        grid = box.grid_flow(columns=2, align=True, row_major=True)
        
        wheel_label_text = "Wheel " + wheel_name
        if is_vehicle_body is True:
            wheel_label_text = "Vehicle Body"

        grid.label(text = wheel_label_text)
        grid.prop(scene, prop_name, text = "")

        if scene.rig_calipers is True and is_vehicle_body is False:
            grid.label(text = "Caliper " + wheel_name)
            grid.prop(scene, prop_name.replace("wheel", "caliper"), text = "")

    def n_wheel_prop(self, item, index, scene):
        box = self.layout.box()
        row = box.row(align=True)
        grid = row.grid_flow(columns=2, align=True, row_major=True)
        grid.prop(item, 'wheel_name', text = "")
        grid.prop(item, 'wheel_mesh', text = "")

        if scene.rig_calipers is True:
            grid.label(text = item.wheel_name + " caliper")
            grid.prop(item, 'caliper_mesh', text = "")
        
        row.alignment = "RIGHT"
        row.operator('view3d.remove_chosen_wheel', text="", icon="TRASH").id = index

class UI_PT_Scene_Setup_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Scene_Setup_Panel"
    bl_label = "Scene Setup"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout

        scene = context.scene

        row = layout.row()
        row.operator('view3d.set_unit_scale')

        row = layout.row()
        row.operator('view3d.reset_unit_scale')

        row = layout.row()
        row.operator('view3d.upscale_objects')

        row = layout.row()
        row.operator('view3d.downscale_objects')

class UI_PT_Additional_Rigging_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Additional_Rigging_Panel"
    bl_label = "Additional Rigging"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        scene = context.scene

        split = layout.split(factor=0.5)
        col1 = split.column(align=True)
        col1.alignment='RIGHT' 
        col2 = split.column(align=True)

        # grid = layout.grid_flow(columns = 2, align = True, row_major = True)
        col1.label(text = "Vehicle Armature")
        col2.prop(scene, 'armature_for_new_bone', text = "")
        
        if scene.armature_for_new_bone is not None:
            col1.label(text = "Parent Bone")
            col2.prop_search(scene, "armature_parent_bone_name", scene.armature_for_new_bone, "bones", icon = "BONE_DATA", text = "")
            
            col1.label(text = "New Bone Name")
            col2.prop(scene, 'new_bone_name', text ="")
        

class UI_PT_Edit_Mode_Bone_Adding_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Edit_Mode_Bone_Adding_Panel"
    bl_parent_id = "UI_PT_Additional_Rigging_Panel"
    bl_label = "Edit Mode Rigging"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {"DEFAULT_CLOSED"}

    def draw(self, context):
        layout = self.layout      
        scene = context.scene 
        grid = layout.grid_flow(columns = 1, align = True)

        row = grid.row()
        row.label(text = "Bone Head")
        grid.prop(scene, 'bone_head_location', text = "")
        grid.operator('view3d.set_bone_head_location', text = "Set Head Location")
        grid.operator('view3d.set_3d_cursor_to_location').new_location = scene.bone_head_location

        row = grid.row()
        row.label(text = "Bone Tail")
        grid.prop(scene, 'bone_tail_location', text = "")
        grid.operator('view3d.set_bone_tail_location', text = "Set Tail Location")
        grid.operator('view3d.set_3d_cursor_to_location').new_location = scene.bone_tail_location

        row = layout.row()
        row.prop(scene, 'set_vertext_groups_to_selected')

        row = layout.row()
        row.prop(scene, 'edit_mode_end_in_pose_mode')
        
        row = layout.row()
        row.operator('view3d.add_bone_to_armature', text = "Add Bone")

class UI_PT_Object_Mode_Bone_Adding_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Object_Mode_Bone_Adding_Panel"
    bl_parent_id = "UI_PT_Additional_Rigging_Panel"
    bl_label = "Object Mode Rigging"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {"DEFAULT_CLOSED"}

    def draw(self, context):
        layout = self.layout      
        scene = context.scene 

        split = layout.split(factor=0.5)
        col1 = split.column(align=True)
        col1.alignment='RIGHT' 
        col2 = split.column(align=True)
        col1.label(text="End Mode:")
        col2.prop(scene.object_mode_end_mode, 'end_mode', expand=False, text="")

        row = layout.row()
        row.operator('view3d.add_bone_to_object')
        row = layout.row()
        row.operator('view3d.add_object_to_rig')

class UI_PT_Mesh_Weight_Troubleshooting_Panel(bpy.types.Panel):
    bl_idname = "UI_PT_Mesh_Weight_Troubleshooting_Panel"
    bl_label = "Mesh Weight Troubleshooting"
    bl_category = "UE Vehicle"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {"DEFAULT_CLOSED"}

    def draw(self, context):
        layout = self.layout      
        scene = context.scene 

        layout.operator('view3d.get_vertex_group_sums')
        row = layout.row(align = True)
        row.label(text="Troubleshoot Object")
        row.prop(scene, 'troubleshoot_object', text= "")
        column = layout.column(align = True)

        for weight in scene.object_vertex_group_weights:
            column.prop(weight, 'value', text = weight.name)

        layout.operator('view3d.update_mesh_weights_from_list')
