# Copyright (C) 2025 Arturs Ontuzans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import bpy, math

class Export_Rig_FBX_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.export_vehicle_rig_fbx"
    bl_label = "FBX"
    bl_description = "Updates rig for FBX export and exports it."
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        scene = bpy.context.scene

        bpy.ops.object.mode_set(mode='OBJECT')
        selected_objects = bpy.context.selected_objects.copy()
        bpy.ops.object.select_all(action='DESELECT')

        armature = None
        for obj in scene.objects:
            if obj.type == 'ARMATURE':
                obj.select_set(True)
                armature = obj
                bpy.context.view_layer.objects.active = obj
                break        

        if armature is None:
            self.report({'ERROR'}, 'No Armature found in scene')
            return {'FINISHED'}
        
        temp = scene.armature_for_new_bone
        scene.armature_for_new_bone = None

        unit_length = bpy.context.scene.unit_settings.scale_length

        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)

        bpy.ops.object.mode_set(mode='EDIT')
        P = context.preferences.addons[__package__].preferences
        bone_length = scene.bone_length

        flip_bone(P.root_bone_name, armature, bone_length, False)

        caliper_prefix = P.caliper_bone_prefix
        caliper_suffix = P.caliper_bone_suffix

        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                wheel_name = wheel_item.wheel_name
                flip_bone(wheel_name, armature, bone_length, False)
                if scene.rig_calipers is True:
                    caliper_name = caliper_prefix + wheel_name + caliper_suffix
                    flip_bone(caliper_name, armature, bone_length, False)
        else:
            flip_bone(P.RL_bone_name, armature, bone_length, False)
            flip_bone(P.RR_bone_name, armature, bone_length, False)
            flip_bone(P.FL_bone_name, armature, bone_length, False)
            flip_bone(P.FR_bone_name, armature, bone_length, False)
            if scene.rig_calipers is True:
                flip_bone(caliper_prefix + P.RL_bone_name + caliper_suffix, armature, bone_length, False)
                flip_bone(caliper_prefix + P.RR_bone_name + caliper_suffix, armature, bone_length, False)
                flip_bone(caliper_prefix + P.FL_bone_name + caliper_suffix, armature, bone_length, False)
                flip_bone(caliper_prefix + P.FR_bone_name + caliper_suffix, armature, bone_length, False)
        bpy.ops.object.mode_set(mode='OBJECT')

        if math.isclose(unit_length, 1, abs_tol=0.001):
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
            armature.scale = (100, 100, 100)
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
            armature.scale = (0.01, 0.01, 0.01)
        
        scene.armature_for_new_bone = temp

        for obj in selected_objects:
            obj.select_set(True)

        bpy.ops.export_scene.fbx(
            'INVOKE_DEFAULT',
            object_types = {'MESH', 'ARMATURE'},
            mesh_smooth_type  = 'FACE',
            add_leaf_bones = False
        )

        return {'FINISHED'}
    
class Export_Rig_GLTF_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.export_vehicle_rig_gltf"
    bl_label = "GLTF"
    bl_description = "Updates rig for GLTF export and exports it."
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        scene = bpy.context.scene

        bpy.ops.object.mode_set(mode='OBJECT')
        selected_objects = bpy.context.selected_objects.copy()
        bpy.ops.object.select_all(action='DESELECT')

        armature = None
        for obj in scene.objects:
            if obj.type == 'ARMATURE':
                obj.select_set(True)
                armature = obj
                bpy.context.view_layer.objects.active = obj
                break        

        if armature is None:
            self.report({'ERROR'}, 'No Armature found in scene')
            return {'FINISHED'}
        
        temp = scene.armature_for_new_bone
        scene.armature_for_new_bone = None

        unit_length = bpy.context.scene.unit_settings.scale_length

        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
        
        if math.isclose(unit_length, 0.01, abs_tol=0.001):
            bone_scale_diff = bpy.context.scene.unit_settings.scale_length / 1
            bpy.context.scene.unit_settings.system = 'METRIC'
            bpy.context.scene.unit_settings.scale_length = 1
            bpy.context.space_data.clip_end = 1000
            bpy.context.space_data.clip_start = 0.01
            armature.scale = (0.01, 0.01, 0.01)
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
            bpy.ops.view3d.view_selected()
            context.scene.bone_length *= bone_scale_diff

        
        scene.armature_for_new_bone = temp

        bpy.ops.object.mode_set(mode='EDIT')

        P = context.preferences.addons[__package__].preferences
        bone_length = scene.bone_length

        flip_bone(P.root_bone_name, armature, bone_length, True)

        caliper_prefix = P.caliper_bone_prefix
        caliper_suffix = P.caliper_bone_suffix

        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                wheel_name = wheel_item.wheel_name
                flip_bone(wheel_name, armature, bone_length, True)
                if scene.rig_calipers is True:
                    caliper_name = caliper_prefix + wheel_name + caliper_suffix
                    flip_bone(caliper_name, armature, bone_length, True)
        else:
            flip_bone(P.RL_bone_name, armature, bone_length, True)
            flip_bone(P.RR_bone_name, armature, bone_length, True)
            flip_bone(P.FL_bone_name, armature, bone_length, True)
            flip_bone(P.FR_bone_name, armature, bone_length, True)
            if scene.rig_calipers is True:
                flip_bone(caliper_prefix + P.RL_bone_name + caliper_suffix, armature, bone_length, True)
                flip_bone(caliper_prefix + P.RR_bone_name + caliper_suffix, armature, bone_length, True)
                flip_bone(caliper_prefix + P.FL_bone_name + caliper_suffix, armature, bone_length, True)
                flip_bone(caliper_prefix + P.FR_bone_name + caliper_suffix, armature, bone_length, True)

        bpy.ops.object.mode_set(mode='OBJECT')

        for obj in selected_objects:
            obj.select_set(True)

        bpy.ops.export_scene.gltf(
            'INVOKE_DEFAULT'
        )

        return {'FINISHED'}

def flip_bone(bone_name, armature, bone_length, up):
    if bone_name not in armature.data.edit_bones: 
        return
    bone = armature.data.edit_bones[bone_name]
    head = bone.head
    tail = bone.tail
    
    if up is True:
        tail.y = head.y
        tail.z = head.z + bone_length
    else:
        tail.y = head.y + bone_length
        tail.z = head.z 