# Copyright (C) 2025 Arturs Ontuzans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import bpy, math
from mathutils import Vector
from . helper_methods import add_bone_for_caliper, add_child_bone, check_and_unlink_objects, get_object_geometry_origin_position
from . helper_methods import clear_old_armature_modifiers, select_object_if_set,set_vertex_group,set_vertex_group_if_set
from . helper_methods import add_child_bone_at_location

class Rig_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.rig_vehicle"
    bl_label = "Rig Vehicle"
    bl_description = "Rigs Vehicle for UE. Unit Scale needs to be set to 0.01 to have it enabled. " \
    "You can set it in the Scene settings or press the \"Set Unit Scale\" button in the addons \"UE Scene Setup\" section."
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        unit_length = bpy.context.scene.unit_settings.scale_length
        unit_system = bpy.context.scene.unit_settings.system
        return (((context.scene.vehicle_body is not None 
            and context.scene.wheel_FR is not None 
            and context.scene.wheel_FL is not None 
            and context.scene.wheel_RR is not None 
            and context.scene.wheel_RL is not None 
            and context.scene.dynamic_wheel_count is False) or (
            context.scene.vehicle_body is not None 
            and context.scene.dynamic_wheel_count is True and 
            all(item.wheel_mesh is not None for item in context.scene.multiple_wheels))) 
#            and math.isclose(unit_length, 0.01, abs_tol=0.001)
            and unit_system == 'METRIC')

    def execute(self, context):
        scene = context.scene

        C = bpy.context 
        D = bpy.data
        O = bpy.ops
        P = context.preferences.addons[__package__].preferences

        prefix = P.caliper_bone_prefix
        suffix = P.caliper_bone_suffix

        #Set variables for vehicle base and wheel meshes
        vehicle_body = scene.vehicle_body
        wheel_RL = scene.wheel_RL
        wheel_RR = scene.wheel_RR
        wheel_FL = scene.wheel_FL
        wheel_FR = scene.wheel_FR
        caliper_RL = scene.caliper_RL
        caliper_RR = scene.caliper_RR
        caliper_FL = scene.caliper_FL
        caliper_FR = scene.caliper_FR

        #Needs some active item to change to object mode
        C.view_layer.objects.active = vehicle_body

        #Set object mode
        O.object.mode_set(mode='OBJECT')

        #Deselect all items
        bpy.ops.object.select_all(action='DESELECT')

        #Set vehicle base as active
        C.view_layer.objects.active = vehicle_body
    
        #Select all vehicle meshes
        vehicle_body.select_set(state=True)
        
        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                wheel_item.wheel_mesh.select_set(state=True)
                select_object_if_set(wheel_item.caliper_mesh)
        else:
            wheel_RL.select_set(state=True)
            wheel_RR.select_set(state=True)
            wheel_FL.select_set(state=True)
            wheel_FR.select_set(state=True)
            if scene.rig_calipers is True:
                select_object_if_set(caliper_RL)
                select_object_if_set(caliper_RR)
                select_object_if_set(caliper_FL)
                select_object_if_set(caliper_FR)

        #Set object mode
        O.object.mode_set(mode='OBJECT')

        #Set all object origins to geometry center
        O.object.origin_set(type='ORIGIN_GEOMETRY', center = 'BOUNDS')

        #making wheel mesh array
        wheel_mesh_array = []
        caliper_mesh_array = []

        if scene.dynamic_wheel_count is True:
            wheel_mesh_array = [wheel_item.wheel_mesh for wheel_item in scene.multiple_wheels]
            caliper_mesh_array = [wheel_item.caliper_mesh for wheel_item in scene.multiple_wheels]
            caliper_mesh_array = list(filter(lambda x: x is not None, caliper_mesh_array))
        else:
            wheel_mesh_array = [wheel_RL, wheel_FL, wheel_RR, wheel_FR]
            if scene.rig_calipers is True:
                caliper_mesh_array = [caliper_RL, caliper_FL, caliper_RR, caliper_FR]
                caliper_mesh_array = list(filter(lambda x: x is not None, caliper_mesh_array))

        #checking for linked meshes and unlinking them
        check_and_unlink_objects(wheel_mesh_array)

        #checking for linked meshes for selecte calipers
        if scene.rig_calipers is True:
            check_and_unlink_objects(caliper_mesh_array)
        
        #remove old armature modifiers from meshes
        clear_old_armature_modifiers(vehicle_body)
        for wheel in wheel_mesh_array:
            clear_old_armature_modifiers(wheel)

        if scene.rig_calipers is True:
            for caliper in caliper_mesh_array:
                clear_old_armature_modifiers(caliper)

        #get all armature type objects
        armature_objects = list(filter(lambda o: o.type == 'ARMATURE', D.objects))

        #remove armature if there's no armature object linked to it
        for armature in D.armatures[:]:
            if not any(x.data == armature for x in armature_objects):
                D.armatures.remove(armature)

        #Apply object transform
        O.object.transform_apply(location = False, rotation = True, scale = True)

        #Create armature object
        armature = D.armatures.new('Armature')
        armature_object = D.objects.new('Armature', armature)

        #Link armature object to our scene
        C.collection.objects.link(armature_object)

        #Make armature variable
        armature_data = D.objects[armature_object.name]

        #Set armature active
        C.view_layer.objects.active = armature_data

        #Set armature selected
        armature_data.select_set(state=True)

        #Set edit mode
        O.object.mode_set(mode='EDIT')

        #Set bones In front and show axis
        armature_data.show_in_front = True
        armature_data.data.show_axes = True

        #Add root bone
        root_bone = armature_data.data.edit_bones.new(P.root_bone_name)
        #Set its orientation and size
        root_bone.head = (0,0,0)
        root_bone.tail = (0,scene.bone_length,0)

        if P.root_bone_name == vehicle_body.name:
            vehicle_body.name = vehicle_body.name + "_mesh"
        
        if scene.body_separate_bone:
            body_bone = armature_data.data.edit_bones.new(P.body_bone_name)
            body_bone.head = (0,0,0)
            body_bone.tail = (0, scene.bone_length, 0)
            body_bone.matrix = vehicle_body.matrix_world
            body_bone.parent = root_bone
            if P.body_bone_name == vehicle_body.name:
                vehicle_body.name = vehicle_body.name + "_mesh"
        else:
            #Set its location to vehicle base mesh
            root_bone.matrix = vehicle_body.matrix_world

        #Add wheel bones to armature
        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                wheel_item.wheel_name = add_child_bone(wheel_item.wheel_name, root_bone, wheel_item.wheel_mesh, armature_data, scene.bone_length).name
                if scene.rig_calipers is True:
                    if scene.wheels_as_parents_of_calipers:
                        add_bone_for_caliper(wheel_item.caliper_mesh, prefix + wheel_item.wheel_name + suffix, armature_data.data.edit_bones[wheel_item.wheel_name], wheel_item.wheel_mesh, armature_data, scene.bone_length)
                    else:
                        add_bone_for_caliper(wheel_item.caliper_mesh, prefix + wheel_item.wheel_name + suffix, root_bone, wheel_item.wheel_mesh, armature_data, scene.bone_length)
        else:
            add_child_bone(P.RL_bone_name, root_bone, wheel_RL, armature_data, scene.bone_length)
            add_child_bone(P.RR_bone_name, root_bone, wheel_RR, armature_data, scene.bone_length)
            add_child_bone(P.FL_bone_name, root_bone, wheel_FL, armature_data, scene.bone_length)
            add_child_bone(P.FR_bone_name, root_bone, wheel_FR, armature_data, scene.bone_length)
            if scene.rig_calipers is True:

                if scene.wheels_as_parents_of_calipers:
                    add_bone_for_caliper(caliper_RL, prefix + P.RL_bone_name + suffix,
                                        armature_data.data.edit_bones[P.RL_bone_name], wheel_RL, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_RR, prefix + P.RR_bone_name + suffix,
                                        armature_data.data.edit_bones[P.RR_bone_name], wheel_RR, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_FL, prefix + P.FL_bone_name + suffix,
                                        armature_data.data.edit_bones[P.FL_bone_name], wheel_FL, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_FR, prefix + P.FR_bone_name + suffix,
                                        armature_data.data.edit_bones[P.FR_bone_name], wheel_FR, armature_data, scene.bone_length)
                else:
                    add_bone_for_caliper(caliper_RL, prefix + P.RL_bone_name + suffix,
                                        root_bone, wheel_RL, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_RR, prefix + P.RR_bone_name + suffix,
                                        root_bone, wheel_RR, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_FL, prefix + P.FL_bone_name + suffix,
                                        root_bone, wheel_FL, armature_data, scene.bone_length)
                    add_bone_for_caliper(caliper_FR, prefix + P.FR_bone_name + suffix,
                                        root_bone, wheel_FR, armature_data, scene.bone_length)

        #Set object mode
        O.object.mode_set(mode='OBJECT')

        #Select vehicle meshes
        vehicle_body.select_set(state=True)

        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                wheel_item.wheel_mesh.select_set(state=True)
        else:
            wheel_RL.select_set(state=True)
            wheel_RR.select_set(state=True)
            wheel_FL.select_set(state=True)
            wheel_FR.select_set(state=True)
            if scene.rig_calipers is True:
                select_object_if_set(caliper_RL)
                select_object_if_set(caliper_RR)
                select_object_if_set(caliper_FL)
                select_object_if_set(caliper_FR)

        #Set armature active
        C.view_layer.objects.active = armature_data

        #Parent meshes to armature with empty groups
        O.object.parent_set(type='ARMATURE_NAME')

        #Set mesh vertex groups/weightpaint mesh
        if scene.body_separate_bone:
            set_vertex_group(vehicle_body, P.body_bone_name)
        else:
            set_vertex_group(vehicle_body, P.root_bone_name)

        if scene.dynamic_wheel_count is True:
            for wheel_item in scene.multiple_wheels:
                set_vertex_group(wheel_item.wheel_mesh, wheel_item.wheel_name)
                set_vertex_group_if_set(wheel_item.caliper_mesh, prefix + wheel_item.wheel_name + suffix)
        else:
            set_vertex_group(wheel_RL, P.RL_bone_name)
            set_vertex_group(wheel_RR, P.RR_bone_name)
            set_vertex_group(wheel_FL, P.FL_bone_name)
            set_vertex_group(wheel_FR, P.FR_bone_name)
            if scene.rig_calipers is True:
                set_vertex_group_if_set(caliper_RL, prefix + P.RL_bone_name + suffix)
                set_vertex_group_if_set(caliper_RR, prefix + P.RR_bone_name + suffix)
                set_vertex_group_if_set(caliper_FL, prefix + P.FL_bone_name + suffix)
                set_vertex_group_if_set(caliper_FR, prefix + P.FR_bone_name + suffix)

        #Deselect all objects
        O.object.select_all(action='DESELECT')
        #Set pose mode
        O.object.mode_set(mode='POSE')

        return {'FINISHED'}


class Scale_Units_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.set_unit_scale"
    bl_label = "Set Unit Scale 0.01"
    bl_description = "Set Scene Unit Scale for UE"
    bl_options = {'REGISTER', 'UNDO'}
    
    @classmethod
    def poll(cls, context):
        unit_length = bpy.context.scene.unit_settings.scale_length
        return not math.isclose(unit_length, 0.01, abs_tol=0.001) or bpy.context.scene.unit_settings.system != 'METRIC'

    def execute(self, context):
        bone_scale_diff = bpy.context.scene.unit_settings.scale_length / 0.01
        bpy.context.scene.unit_settings.system = 'METRIC'
        bpy.context.scene.unit_settings.scale_length = 0.01
        bpy.context.space_data.clip_end = 100000
        bpy.context.space_data.clip_start = 1
        context.scene.bone_length *= bone_scale_diff

        return {'FINISHED'}
    
class Reset_Units_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.reset_unit_scale"
    bl_label = "Reset Unit Scale"
    bl_description = "Reset Unit Scale for UE"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        bone_scale_diff = bpy.context.scene.unit_settings.scale_length / 1
        bpy.context.scene.unit_settings.system = 'METRIC'
        bpy.context.scene.unit_settings.scale_length = 1
        bpy.context.space_data.clip_end = 1000
        bpy.context.space_data.clip_start = 0.01
        context.scene.bone_length *= bone_scale_diff

        return {'FINISHED'}


class Upscale_Objects_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.upscale_objects"
    bl_label = "Upscale Objects 100x"
    bl_description = "Set Scene Unit Scale for UE"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        if bpy.context.active_object is None:
            if bpy.context.scene.objects:
                bpy.context.view_layer.objects.active = bpy.context.scene.objects[0]

        bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.view3d.snap_cursor_to_center()
        bpy.ops.object.select_all(action='SELECT')

        bpy.context.scene.tool_settings.transform_pivot_point = 'CURSOR'
        bpy.context.scene.cursor.location = (0, 0, 0)

        bpy.ops.transform.resize(value=(100, 100, 100), orient_type='GLOBAL', 
            orient_matrix=((1, 0, 0), (0, 1, 0), (0, 0, 1)), 
            orient_matrix_type='GLOBAL', mirror=True, use_proportional_edit=False, 
            proportional_edit_falloff='SMOOTH', proportional_size=1, 
            use_proportional_connected=False, use_proportional_projected=False)

        check_and_unlink_objects(bpy.context.selected_objects)

        bpy.ops.object.transform_apply(location = False, rotation = True, scale = True)

        bpy.context.scene.tool_settings.transform_pivot_point = 'MEDIAN_POINT'

        bpy.ops.view3d.view_selected()
        
        return {'FINISHED'}
    
class Downscale_Objects_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.downscale_objects"
    bl_label = "Downscale Objects 100x"
    bl_description = "Set Scene Unit Scale for UE"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        if bpy.context.active_object is None:
            if bpy.context.scene.objects:
                bpy.context.view_layer.objects.active = bpy.context.scene.objects[0]

        bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.view3d.snap_cursor_to_center()
        bpy.ops.object.select_all(action='SELECT')

        bpy.context.scene.tool_settings.transform_pivot_point = 'CURSOR'
        bpy.context.scene.cursor.location = (0, 0, 0)

        bpy.ops.transform.resize(value=(0.01, 0.01, 0.01), orient_type='GLOBAL', 
            orient_matrix=((1, 0, 0), (0, 1, 0), (0, 0, 1)), 
            orient_matrix_type='GLOBAL', mirror=True, use_proportional_edit=False, 
            proportional_edit_falloff='SMOOTH', proportional_size=1, 
            use_proportional_connected=False, use_proportional_projected=False)

        check_and_unlink_objects(bpy.context.selected_objects)

        bpy.ops.object.transform_apply(location = False, rotation = True, scale = True)

        bpy.context.scene.tool_settings.transform_pivot_point = 'MEDIAN_POINT'

        bpy.ops.view3d.view_selected()
        
        return {'FINISHED'}

class Set_Bone_Head_Location_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.set_bone_head_location"
    bl_label = "Set Head Location"
    bl_description = "Set Bone Head Location to selection"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        scene = context.scene
        #TODO check if selected mesh not armature
        if context.active_object.mode == 'EDIT':
            obj = bpy.context.edit_object
            obj.update_from_editmode()
            me = obj.data

            mat = obj.matrix_world

            verts_sel = [v.co for v in me.vertices if v.select]

            loc = mat @ (sum(verts_sel, Vector()) / len(verts_sel))
            scene.bone_head_location = loc
    
        return {'FINISHED'}


class Set_Bone_Tail_Location_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.set_bone_tail_location"
    bl_label = "Set Tail Location"
    bl_description = "Set Bone Tail Location to selection"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        scene = context.scene
        #TODO check if selected mesh not armature
        if context.active_object.mode == 'EDIT':    
            obj = bpy.context.edit_object
            obj.update_from_editmode()
            me = obj.data

            mat = obj.matrix_world

            verts_sel = [v.co for v in me.vertices if v.select]

            loc = mat @ (sum(verts_sel, Vector()) / len(verts_sel))
            scene.bone_tail_location = loc
    
        return {'FINISHED'}


class Add_Bone_To_Armature_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.add_bone_to_armature"
    bl_label = "Add Bone To Armature"
    bl_description = "Add Bone To Selected Armature. Only usable in edit mode, and the Parent Bone needs to be set."
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        parent_name = context.scene.armature_parent_bone_name
        armature_data = context.scene.armature_for_new_bone
        return (len(parent_name) > 0 and 
            context.object is not None and 
            context.object.mode == "EDIT" and 
            context.object.type == "MESH" and 
            armature_data is not None)

    def execute(self, context):
        C = bpy.context 
        D = bpy.data
        O = bpy.ops
        scene = context.scene

        armature_data = scene.armature_for_new_bone
        parent_name = scene.armature_parent_bone_name
        
        arm_obj = next(filter(lambda o: o.type == 'ARMATURE' and o.data == armature_data, D.objects))


        selected_object = bpy.context.selected_objects[0]

        #Need to have object mode so selected vertices would get updated
        O.object.mode_set(mode='OBJECT')

        selected_vertice_indexes = [v.index for v in selected_object.data.vertices if v.select]

        C.view_layer.objects.active = arm_obj
        arm_obj.select_set(state=True)

        #Need to reset scale to 0.01 if it got set up for FBX export without unit scale change
        #For some reason if PointerProperty is set, it sees it as another user and it doesn't allow to do transform
        scene.armature_for_new_bone = None
        O.object.transform_apply(location = False, rotation = False, scale = True)
        scene.armature_for_new_bone = armature_data

        O.object.parent_set(type='ARMATURE_NAME')

        O.object.mode_set(mode='EDIT')

        parent_bone = armature_data.edit_bones[parent_name]

        new_bone_name = scene.new_bone_name

        new_bone = armature_data.edit_bones.new(new_bone_name)
        new_bone.head = scene.bone_head_location
        new_bone.tail = scene.bone_tail_location
        new_bone.parent = parent_bone
        selected_object.vertex_groups.new(name = new_bone.name)
        
        if scene.set_vertext_groups_to_selected:
            arm_obj.select_set(state=False)
            selected_object.select_set(state=True)
            C.view_layer.objects.active  = selected_object
            O.object.mode_set(mode='OBJECT', toggle = True)

            for group in selected_object.vertex_groups:
                selected_object.vertex_groups[group.name].add(selected_vertice_indexes, 1, 'SUBTRACT')

            selected_object.vertex_groups[new_bone.name].add(selected_vertice_indexes, 1, 'REPLACE')
            selected_object.select_set(state=False)

        if not scene.edit_mode_end_in_pose_mode:
            arm_obj.select_set(state=False)
            selected_object.select_set(state=True)
            #Need to go into armature mode so it would get visually updated. Tweek around, maybe can find better solution
            C.view_layer.objects.active = arm_obj
            O.object.mode_set(mode='POSE', toggle = True)

            C.view_layer.objects.active = selected_object
            O.object.mode_set(mode='EDIT', toggle = True)
        else:
            C.view_layer.objects.active = arm_obj
            O.object.mode_set(mode='POSE', toggle = True)

        return {'FINISHED'}

class Add_Bone_To_Object_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.add_bone_to_object"
    bl_label = "Create Bone and Add"
    bl_description = "Creates a new bone for selected objects and then adds it and the selected objects to the chosen \"Vehicle Armature\". " \
    "Selected objects get rigged to the newly created bone. You can set the new bone name in the \"New Bone Name\" field above. Usable only in the Object Mode."
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        selected_object = context.active_object
        armature = context.scene.armature_for_new_bone
        parent_bone = context.scene.armature_parent_bone_name
        return (selected_object is not None and selected_object.type == 'MESH' 
                and context.object.mode == "OBJECT"
                and selected_object.select_get() 
                and armature is not None and len(parent_bone) > 0)

    def execute(self, context):
        O = bpy.ops
        obj = context.object
        mesh_position = get_object_geometry_origin_position(obj)

        check_and_unlink_objects([obj])
        clear_old_armature_modifiers(obj)

        O.object.transform_apply(location = False, rotation = True, scale = True)

        armature_data = context.scene.armature_for_new_bone
        parent_name = context.scene.armature_parent_bone_name
        
        arm_obj = next(filter(lambda o: o.type == 'ARMATURE' and o.data == armature_data, bpy.data.objects))

        context.view_layer.objects.active = arm_obj
        arm_obj.select_set(state=True)

        #Need to reset scale to 0.01 if it got set up for FBX export without unit scale change
        #For some reason if PointerProperty is set, it sees it as another user and it doesn't allow to do transform
        context.scene.armature_for_new_bone = None
        O.object.transform_apply(location = False, rotation = False, scale = True)
        context.scene.armature_for_new_bone = armature_data

        O.object.mode_set(mode='EDIT')

        parent_bone = armature_data.edit_bones[parent_name]
        new_bone_name = context.scene.new_bone_name

        new_bone_name = add_child_bone_at_location(new_bone_name, parent_bone, mesh_position, arm_obj, context.scene.bone_length).name
        O.object.mode_set(mode='OBJECT')

        obj.select_set(state = True)
        context.view_layer.objects.active = arm_obj

        O.object.parent_set(type='ARMATURE_NAME')
        
        selected_mesh_objects = [obj for obj in bpy.context.selected_objects if obj.type == 'MESH']
        for obj in selected_mesh_objects:
            set_vertex_group(obj, new_bone_name)

        end_mode = context.scene.object_mode_end_mode.end_mode 

        if end_mode == "OBJECT MODE":
            O.object.select_all(action='DESELECT')

        if end_mode == "EDIT MODE":
            O.object.select_all(action='DESELECT')
            O.object.mode_set(mode='EDIT')

            new_edit_bone = context.object.data.edit_bones[new_bone_name]
            new_edit_bone.select = True
            new_edit_bone.select_head = True
            new_edit_bone.select_tail = True
            context.object.data.edit_bones.active = new_edit_bone

        if end_mode == "POSE MODE":
            O.object.select_all(action='DESELECT')
            context.view_layer.objects.active = arm_obj
            arm_obj.select_set(state=True)
            O.object.mode_set(mode='POSE', toggle = True)
            new_bone = arm_obj.pose.bones[new_bone_name]

            new_bone.select = True
            arm_obj.data.bones.active = new_bone.bone
        
        return {'FINISHED'}
    
class Add_Object_To_Rig_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.add_object_to_rig"
    bl_label = "Just Add to Bone"
    bl_description = "Adds selected objects to the chosen \"Vehicle Armature\" and rigs to the selected \"Parent Bone\". Usable only in the Object Mode."
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        selected_object = context.active_object
        armature = context.scene.armature_for_new_bone
        parent_bone = context.scene.armature_parent_bone_name
        return (selected_object is not None and selected_object.type == 'MESH' 
                and context.object.mode == "OBJECT"
                and selected_object.select_get()
                and armature is not None and len(parent_bone) > 0)

    def execute(self, context):
        O = bpy.ops
        obj = context.object
        O.object.transform_apply(location = False, rotation = False, scale = True)

        armature_data = context.scene.armature_for_new_bone
        parent_name = context.scene.armature_parent_bone_name

        arm_obj = next(filter(lambda o: o.type == 'ARMATURE' and o.data == armature_data, bpy.data.objects))

        context.view_layer.objects.active = arm_obj
        arm_obj.select_set(state=True)
        obj.select_set(state = True)
        
        #Need to reset scale to 0.01 if it got set up for FBX export without unit scale change
        #For some reason if PointerProperty is set, it sees it as another user and it doesn't allow to do transform
        context.scene.armature_for_new_bone = None
        O.object.transform_apply(location = False, rotation = False, scale = True)
        context.scene.armature_for_new_bone = armature_data

        O.object.parent_set(type='ARMATURE_NAME')
        
        selected_mesh_objects = [obj for obj in bpy.context.selected_objects if obj.type == 'MESH']
        for obj in selected_mesh_objects:
            set_vertex_group(obj, parent_name)
        
        end_mode = context.scene.object_mode_end_mode.end_mode 

        if end_mode == "OBJECT MODE":
            O.object.select_all(action='DESELECT')

        if end_mode == "EDIT MODE":
            O.object.select_all(action='DESELECT')
            O.object.mode_set(mode='EDIT')

            parent_bone = context.object.data.edit_bones[parent_name]
            parent_bone.select = True
            parent_bone.select_head = True
            parent_bone.select_tail = True
            context.object.data.edit_bones.active = parent_bone

        if end_mode == "POSE MODE":
            O.object.select_all(action='DESELECT')

            context.view_layer.objects.active = arm_obj
            arm_obj.select_set(state=True)
            O.object.mode_set(mode='POSE', toggle = True)
            parent_bone = arm_obj.pose.bones[parent_name]

            parent_bone.select = True
            arm_obj.data.bones.active = parent_bone.bone

        return {'FINISHED'}

class Add_Another_Wheel_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.add_another_wheel"
    bl_label = "Add Another Wheel"
    bl_description = "Add Another Wheel Option"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        P = context.preferences.addons[__package__].preferences
        new = context.scene.multiple_wheels.add()
        new.wheel_name = P.N_bone_name + "." + str(len(context.scene.multiple_wheels))

        return {'FINISHED'}


class Remove_Chosen_Wheel_OT_Operator(bpy.types.Operator):
    bl_idname = "view3d.remove_chosen_wheel"
    bl_label = "Remove Wheel"
    bl_description = "Removes The Wheel from the List"
    bl_options = {'REGISTER', 'UNDO'}

    id: bpy.props.IntProperty()

    def execute(self, context):
        # item = context.scene.multiple_wheels[len(context.scene.multiple_wheels)-1]
        context.scene.multiple_wheels.remove(self.id)

        return {'FINISHED'}