# Copyright (C) 2025 Arturs Ontuzans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import bpy, mathutils
from collections import defaultdict

def check_and_unlink_objects(object_array):
    for index, mesh in enumerate(object_array):
        for other_mesh in object_array[index+1:]:
            if mesh.data == other_mesh.data:
                other_mesh.data = mesh.data.copy()

def clear_old_armature_modifiers(mesh):
        for modifier in mesh.modifiers:
            if modifier.type == 'ARMATURE':
                mesh.modifiers.remove(modifier)

def set_vertex_group(mesh, vertex_group):
    #Make a new list for all vertice indexes
    index_list = [v.index for v in mesh.data.vertices]
    #Populate index list with vertice indexes
    mesh.data.vertices.foreach_get('index', index_list)   
    #Set mesh vertex group
    mesh.vertex_groups[vertex_group].add(index_list, 1, 'REPLACE')
    #Make all vertex group array
    all_vertex_groups = mesh.vertex_groups.keys()
    #Remove group which will not be removed
    all_vertex_groups.remove(vertex_group)
    #Remove unneeded vertex groups
    for group in all_vertex_groups:
        mesh.vertex_groups[group].add(index_list, 1, 'SUBTRACT')

    
def add_child_bone(bone_name, parent_bone, wheel_mesh, armature_data, bone_length):
    #Create a new bone
    new_bone = armature_data.data.edit_bones.new(bone_name)
    #Set bone's size
    new_bone.head = (0,0,0)
    new_bone.tail = (0,bone_length,0)
    #Set bone's parent
    new_bone.parent = parent_bone
    #Set bone's location to wheel
    new_bone.matrix = wheel_mesh.matrix_world

    if bone_name == wheel_mesh.name:
        wheel_mesh.name = wheel_mesh.name + "_mesh"

    return new_bone

def add_child_bone_at_location(bone_name, parent_bone, location, armature_data, bone_length):
    #Create a new bone
    new_bone = armature_data.data.edit_bones.new(bone_name)
    #Set bone's size
    new_bone.head = location.xyz
    tail_location = location.xyz + mathutils.Vector((0, bone_length, 0))
    new_bone.tail = tail_location
    #Set bone's parent
    new_bone.parent = parent_bone
    #Set bone's location to wheel
    return new_bone

def add_bone_for_caliper(caliper_object, bone_name, parent_bone, wheel_mesh, armature_data, bone_length):
    if caliper_object is not None:
        add_child_bone(bone_name, parent_bone, wheel_mesh, armature_data, bone_length)

def select_object_if_set(object):
    if object is not None:
            object.select_set(state=True) 

def set_vertex_group_if_set(object, vertex_group):
    if object is not None:
        set_vertex_group(object, vertex_group)

def get_object_geometry_origin_position(object):
    O = bpy.ops
    C = bpy.context

    cursor_start_location = C.scene.cursor.location.xyz
    C.scene.cursor.location = object.location.xyz

    O.object.origin_set(type = 'ORIGIN_GEOMETRY', center = 'BOUNDS')
    actual_object_position = object.location.xyz
    O.object.origin_set(type = 'ORIGIN_CURSOR')

    C.scene.cursor.location = cursor_start_location

    return actual_object_position

def show_object_vertext_groups(object):
    if object is None:
        bpy.context.scene.object_vertex_group_weights.clear()
        return

    vertex_groups = object.vertex_groups
    group_weights = defaultdict(float)
    bpy.context.scene.object_vertex_group_weights.clear()
    # Iterate through all vertices
    for v in object.data.vertices:
        for g in v.groups:
            group_name = vertex_groups[g.group].name  # Get group name
            group_weights[group_name] += g.weight    # Sum weights
    for g in object.vertex_groups:
        group_weights[g.name] += 0

    # print("Vertex Group Influence:")
    for group, weight in group_weights.items():
        new = bpy.context.scene.object_vertex_group_weights.add()
        new.name = group
        new.value = weight

def on_troubleshoot_object_update(self, context):
    obj = context.scene.troubleshoot_object
    show_object_vertext_groups(obj)