# Copyright (C) 2025 Arturs Ontuzans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

bl_info = {
    "name" : "Unreal Engine Vehicle Rigger",
    "author" : "Arturs Ontuzans",
    "description" : "Helps to rig vehicles for Unreal Engine 4/5.",
    "blender" : (5, 0, 0),
    "version" : (0, 6, 967),
    "location" : "View3D",
    "warning" : "",
    "category" : "Generic"
}

import bpy

from . rig_op import Rig_OT_Operator, Scale_Units_OT_Operator, Reset_Units_OT_Operator, Upscale_Objects_OT_Operator, Downscale_Objects_OT_Operator
from . rig_op import Set_Bone_Head_Location_OT_Operator, Set_Bone_Tail_Location_OT_Operator, Add_Bone_To_Armature_OT_Operator
from . rig_op import Add_Another_Wheel_OT_Operator, Remove_Chosen_Wheel_OT_Operator
from . rig_op import Add_Bone_To_Object_OT_Operator, Add_Object_To_Rig_OT_Operator

from . ui_panel import UI_PT_Rig_Panel, UI_PT_Scene_Setup_Panel, UI_PT_Additional_Rigging_Panel
from . ui_panel import UI_PT_Edit_Mode_Bone_Adding_Panel, UI_PT_Object_Mode_Bone_Adding_Panel
from . ui_panel import UI_PT_Mesh_Weight_Troubleshooting_Panel
from . additional_rigging_tools import Get_Mesh_Vertex_Group_Weight_Sums_OT_Operator, Update_Mesh_Weights_From_List_OT_Operator, Set_3D_Cursor_To_Location_OT_Operator

from . exporting_op import Export_Rig_FBX_OT_Operator, Export_Rig_GLTF_OT_Operator

from . preferences import UERiggingAddonPreferences, Reset_Preferences_Property_OT_Operator, Open_Link_OT_Operator

from . helper_methods import on_troubleshoot_object_update

def object_search_poll(self, object):
    return object.type in ['MESH', 'CURVE']

class WheelItem(bpy.types.PropertyGroup):
    wheel_name: bpy.props.StringProperty()
    wheel_mesh: bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, description = "Wheel mesh")
    caliper_mesh: bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, description = "Caliper/Mudguard mesh")

class VertexGroupWeight(bpy.types.PropertyGroup):
    name: bpy.props.StringProperty()
    value: bpy.props.FloatProperty(name="Value", default=0.0, min=0.0, max=1.0, subtype="FACTOR", precision=3)

class ObjectModeEnd(bpy.types.PropertyGroup):
    end_mode: bpy.props.EnumProperty(
        name="End Mode",
        description="In what mode end after pressing button",
        items=[
            ('OBJECT MODE', 'Object Mode', 'Object Mode', '', 0),
            ('POSE MODE', 'Pose Mode', 'Pose Mode', '', 1),
            ('EDIT MODE', 'Edit Mode', 'Edit Mode', '', 2),
        ],
        default='OBJECT MODE'
    )

def register():
    bpy.utils.register_class(WheelItem)
    bpy.utils.register_class(VertexGroupWeight)
    bpy.utils.register_class(ObjectModeEnd)
    bpy.utils.register_class(Rig_OT_Operator)
    bpy.utils.register_class(Scale_Units_OT_Operator)
    bpy.utils.register_class(Reset_Units_OT_Operator)
    bpy.utils.register_class(Upscale_Objects_OT_Operator)
    bpy.utils.register_class(Downscale_Objects_OT_Operator)
    bpy.utils.register_class(Set_Bone_Head_Location_OT_Operator)
    bpy.utils.register_class(Set_Bone_Tail_Location_OT_Operator)
    bpy.utils.register_class(Add_Bone_To_Armature_OT_Operator)
    bpy.utils.register_class(Add_Another_Wheel_OT_Operator)
    bpy.utils.register_class(Remove_Chosen_Wheel_OT_Operator)
    bpy.utils.register_class(Get_Mesh_Vertex_Group_Weight_Sums_OT_Operator)
    bpy.utils.register_class(Update_Mesh_Weights_From_List_OT_Operator)
    bpy.utils.register_class(Add_Bone_To_Object_OT_Operator)
    bpy.utils.register_class(Add_Object_To_Rig_OT_Operator)
    bpy.utils.register_class(UI_PT_Rig_Panel)
    bpy.utils.register_class(UI_PT_Additional_Rigging_Panel)
    bpy.utils.register_class(UI_PT_Edit_Mode_Bone_Adding_Panel)
    bpy.utils.register_class(UI_PT_Object_Mode_Bone_Adding_Panel)
    bpy.utils.register_class(UI_PT_Mesh_Weight_Troubleshooting_Panel)
    bpy.utils.register_class(UI_PT_Scene_Setup_Panel)
    bpy.utils.register_class(UERiggingAddonPreferences)
    bpy.utils.register_class(Reset_Preferences_Property_OT_Operator)
    bpy.utils.register_class(Open_Link_OT_Operator)
    bpy.utils.register_class(Export_Rig_FBX_OT_Operator)
    bpy.utils.register_class(Export_Rig_GLTF_OT_Operator)
    bpy.utils.register_class(Set_3D_Cursor_To_Location_OT_Operator)
    bpy.types.Scene.vehicle_body = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Vehicle Body mesh", description = "Vehicle Body mesh")
    bpy.types.Scene.wheel_FR = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Wheel FR", description = "Front Right Vehicle Wheel mesh")
    bpy.types.Scene.wheel_FL = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Wheel FL", description = "Front Left Vehicle Wheel mesh")
    bpy.types.Scene.wheel_RR = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Wheel RR", description = "Rear Right Vehicle Wheel mesh")
    bpy.types.Scene.wheel_RL = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Wheel RL", description = "Rear Left Vehicle Wheel mesh")
    bpy.types.Scene.caliper_FR = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Caliper FR", description = "Front Right Vehicle Wheel's Caliper mesh")
    bpy.types.Scene.caliper_FL = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Caliper FL", description = "Front Left Vehicle Wheel's Caliper mesh")
    bpy.types.Scene.caliper_RR = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Caliper RR", description = "Rear Right Vehicle Wheel's Caliper mesh")
    bpy.types.Scene.caliper_RL = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, name= "Caliper RL", description = "Rear Left Vehicle Wheel's Caliper mesh")
    bpy.types.Scene.bone_length = bpy.props.FloatProperty(name = "Bone Length", description = "Length of the bones that will be added by the add-on.", default =  0.5, unit = 'LENGTH')
    bpy.types.Scene.bone_head_location = bpy.props.FloatVectorProperty(subtype = 'XYZ', unit = 'LENGTH')
    bpy.types.Scene.bone_tail_location = bpy.props.FloatVectorProperty(subtype = 'XYZ', unit = 'LENGTH')
    #TODO maybe rename armature_for_new_bone variable all together, add also object thing for description
    bpy.types.Scene.armature_for_new_bone = bpy.props.PointerProperty(type=bpy.types.Armature, description = "Armature to which new bone will be added.",   
        name="Bone armature")
    #TODO maybe rename armature_parent_bone_name variable all together, add also object thing for description
    bpy.types.Scene.armature_parent_bone_name = bpy.props.StringProperty(description = "Parent bone of new bone.", name="Parent bone name")
    #TODO add something about new object things too
    bpy.types.Scene.new_bone_name = bpy.props.StringProperty(description = "New bone name", name = "New bone name")
    bpy.types.Scene.set_vertext_groups_to_selected = bpy.props.BoolProperty(default = True, 
        description = "If checked, it will automatically assign selected vertices to a new bone vertex group.", name = "Auto Weight Selected")
    bpy.types.Scene.edit_mode_end_in_pose_mode = bpy.props.BoolProperty(default = False, 
        description = "After adding bone, you will end up in pose mode to check bone weights.", name = "End In Pose Mode")
    bpy.types.Scene.object_mode_end_mode = bpy.props.PointerProperty(type = ObjectModeEnd)
    bpy.types.Scene.multiple_wheels = bpy.props.CollectionProperty(type = WheelItem)
    bpy.types.Scene.dynamic_wheel_count = bpy.props.BoolProperty(default = False, 
        description = "Allows to rig vehicles with more or fewer than 4 wheels.", name = "N-Wheeled vehicle")
    bpy.types.Scene.rig_calipers = bpy.props.BoolProperty(default = False, 
        description = "Adds option to select calipers/mudguards for rigging.", name = "Rig Calipers/Mudguards")
    bpy.types.Scene.wheels_as_parents_of_calipers = bpy.props.BoolProperty(default = True, 
        description = "If True, calipers/mudguards’ parents will be wheel bones. If False, it will be the root bone.", name = "Parent Calipers to Wheels")
    #TODO Change description and naming for body_separate_bone
    bpy.types.Scene.body_separate_bone = bpy.props.BoolProperty(default = False, 
        description = "Will make a separate bone for the vehicle body instead of rigging it to the root bone.", name = "Separate Bone for Vehicle Body")
    bpy.types.Scene.object_vertex_group_weights = bpy.props.CollectionProperty(type = VertexGroupWeight)
    bpy.types.Scene.troubleshoot_object = bpy.props.PointerProperty(type=bpy.types.Object, poll = object_search_poll, update = on_troubleshoot_object_update,
                                                                    name= "Troubleshoot Object", description = "Object shown for weight troubleshooting.", )

def unregister():
    bpy.utils.unregister_class(WheelItem)
    bpy.utils.unregister_class(VertexGroupWeight)
    bpy.utils.unregister_class(ObjectModeEnd)
    bpy.utils.unregister_class(Rig_OT_Operator)
    bpy.utils.unregister_class(Scale_Units_OT_Operator)
    bpy.utils.unregister_class(Reset_Units_OT_Operator)
    bpy.utils.unregister_class(Upscale_Objects_OT_Operator)
    bpy.utils.unregister_class(Downscale_Objects_OT_Operator)
    bpy.utils.unregister_class(Set_Bone_Head_Location_OT_Operator)
    bpy.utils.unregister_class(Set_Bone_Tail_Location_OT_Operator)
    bpy.utils.unregister_class(Add_Bone_To_Armature_OT_Operator)
    bpy.utils.unregister_class(Add_Another_Wheel_OT_Operator)
    bpy.utils.unregister_class(Remove_Chosen_Wheel_OT_Operator)
    bpy.utils.unregister_class(Get_Mesh_Vertex_Group_Weight_Sums_OT_Operator)
    bpy.utils.unregister_class(Update_Mesh_Weights_From_List_OT_Operator)
    bpy.utils.unregister_class(Add_Bone_To_Object_OT_Operator)
    bpy.utils.unregister_class(Add_Object_To_Rig_OT_Operator)
    bpy.utils.unregister_class(UI_PT_Rig_Panel)
    bpy.utils.unregister_class(UI_PT_Additional_Rigging_Panel)
    bpy.utils.unregister_class(UI_PT_Edit_Mode_Bone_Adding_Panel)
    bpy.utils.unregister_class(UI_PT_Object_Mode_Bone_Adding_Panel)
    bpy.utils.unregister_class(UI_PT_Mesh_Weight_Troubleshooting_Panel)
    bpy.utils.unregister_class(UI_PT_Scene_Setup_Panel)
    bpy.utils.unregister_class(UERiggingAddonPreferences)
    bpy.utils.unregister_class(Reset_Preferences_Property_OT_Operator)
    bpy.utils.unregister_class(Open_Link_OT_Operator)
    bpy.utils.unregister_class(Export_Rig_FBX_OT_Operator)
    bpy.utils.unregister_class(Export_Rig_GLTF_OT_Operator)
    bpy.utils.unregister_class(Set_3D_Cursor_To_Location_OT_Operator)
    del bpy.types.Scene.vehicle_body
    del bpy.types.Scene.wheel_FR
    del bpy.types.Scene.wheel_FL
    del bpy.types.Scene.wheel_RR
    del bpy.types.Scene.wheel_RL
    del bpy.types.Scene.bone_length
    del bpy.types.Scene.bone_head_location
    del bpy.types.Scene.bone_tail_location
    del bpy.types.Scene.armature_for_new_bone
    del bpy.types.Scene.armature_parent_bone_name
    del bpy.types.Scene.new_bone_name
    del bpy.types.Scene.set_vertext_groups_to_selected
    del bpy.types.Scene.edit_mode_end_in_pose_mode
    del bpy.types.Scene.object_mode_end_mode
    del bpy.types.Scene.multiple_wheels
    del bpy.types.Scene.dynamic_wheel_count
    del bpy.types.Scene.body_separate_bone
    del bpy.types.Scene.object_vertex_group_weights
    del bpy.types.Scene.troubleshoot_object
